/*----------------------------------------------------------------------------
 * Name:    serial.c
 * Purpose: serial port handling for MCB2300
 * Version: V1.10
 *----------------------------------------------------------------------------
 * This file is part of the uVision/ARM development tools.
 * This software may only be used under the terms of a valid, current,
 * end user licence from KEIL for a compatible version of KEIL software
 * development tools. Nothing else gives you the right to use this software.
 *
 * This software is supplied "AS IS" without warranties of any kind.
 *
 * Copyright (c) 2008 Keil - An ARM Company. All rights reserved.
 *----------------------------------------------------------------------------
 * History:
 *          V1.00 Initial Version
 *          V1.10 Buffer handling added
 *----------------------------------------------------------------------------*/
 
#include <AT91SAM7X256.H>                                     // AT91SAM7X256 definitions
#include "serial.h"

#define UART_CLK               (48000000UL)                   // UART Clock is 48.0 MHz

AT91S_USART *pUSART = AT91C_BASE_US0;                         // Global Pointer to US0

/*----------------------------------------------------------------------------
  Defines for ring buffers
 *---------------------------------------------------------------------------*/
#define SER_BUF_SIZE               (128)                      // serial buffer in bytes (power 2)
#define SER_BUF_MASK               (SER_BUF_SIZE-1ul)         // buffer size mask

// Buffer read / write macros
#define SER_BUF_RESET(serBuf)      (serBuf.rdIdx = serBuf.wrIdx = 0)
#define SER_BUF_WR(serBuf, dataIn) (serBuf.data[SER_BUF_MASK & serBuf.wrIdx++] = (dataIn))
#define SER_BUF_RD(serBuf)         (serBuf.data[SER_BUF_MASK & serBuf.rdIdx++])   
#define SER_BUF_EMPTY(serBuf)      (serBuf.rdIdx == serBuf.wrIdx)
#define SER_BUF_FULL(serBuf)       (serBuf.rdIdx == serBuf.wrIdx+1)
#define SER_BUF_COUNT(serBuf)      (SER_BUF_MASK & (serBuf.wrIdx - serBuf.rdIdx))

// buffer type
typedef struct __SER_BUF_T {
  unsigned char data[SER_BUF_SIZE];
  unsigned int wrIdx;
  unsigned int rdIdx;
} SER_BUF_T;

unsigned long          ser_txRestart;                         // NZ if TX restart is required
unsigned int           ser_lineState;                         // ((msr << 8) | (lsr))
SER_BUF_T              ser_out;                               // Serial data buffers
SER_BUF_T              ser_in;


/*----------------------------------------------------------------------------
  Function Prototypes
 *---------------------------------------------------------------------------*/
static __irq void ser_irq_0 (void);


/*----------------------------------------------------------------------------
  open the serial port
 *---------------------------------------------------------------------------*/
void ser_OpenPort (void) {

  AT91C_BASE_PIOA->PIO_ASR = AT91C_PA0_RXD0 | AT91C_PA1_TXD0; // select A register
  AT91C_BASE_PIOA->PIO_PDR = AT91C_PA0_RXD0 | AT91C_PA1_TXD0; // Set in periph mode 
  AT91C_BASE_PMC->PMC_PCER = 1 << AT91C_ID_US0;               // enable periph clock
}

/*----------------------------------------------------------------------------
  close the serial port
 *---------------------------------------------------------------------------*/
void ser_ClosePort (void) {

  pUSART->US_IDR = AT91C_US_RXRDY | AT91C_US_TXRDY;           // Disable UART interrupts
  AT91C_BASE_AIC->AIC_IECR = (1 << AT91C_ID_US0);             // Disable VIC interrupt
}

/*----------------------------------------------------------------------------
  initialize the serial port
 *---------------------------------------------------------------------------*/
void ser_InitPort (unsigned long baudrate, unsigned int  databits,
                  unsigned int  parity,   unsigned int  stopbits) {

  unsigned long lcr_p, lcr_s, lcr_d;
  unsigned long brg;

  switch (databits) {
    case 5:                                                   // 5 Data bits
      lcr_d = 0x00000000;
    break;
    case 6:                                                   // 6 Data bits
      lcr_d = 0x00000040;
    break;
    case 7:                                                   // 7 Data bits
      lcr_d = 0x00000080;
    break;
    case 8:                                                   // 8 Data bits
    default:
      lcr_d = 0x000000C0;
    break;
  }

  switch (stopbits) {
    case 1:                                                   // 1,5 Stop bits
      lcr_s = 0x00001000;
    case 2:                                                   // 2   Stop bits
      lcr_s = 0x00002000;
    break;
    case 0:                                                   // 1   Stop bit
    default:
      lcr_s = 0x00000000;
    break;
  }

  switch (parity) {
    case 1:                                                   // Parity Odd
      lcr_p = 0x00000200;
    break;
    case 2:                                                   // Parity Even
      lcr_p = 0x00000000;
    break;
    case 3:                                                   // Parity Mark
      lcr_p = 0x00000600;
    break;
    case 4:                                                   // Parity Space
      lcr_p = 0x00000400;
    break;
    case 0:                                                   // Parity None
    default:
      lcr_p = 0x00000800;
    break;
  }

  SER_BUF_RESET(ser_out);                                     // reset out buffer
  SER_BUF_RESET(ser_in);                                      // reset in buffer
  
  // Note that the mclk is 48,0 MHz.
  // 48 MHz MCLK generates also rates for 115200, 57600 baud
  brg = ((UART_CLK * 10UL) / (baudrate * 16UL));
  brg = ((brg % 10) >= 5) ? brg / 10 + 1 : brg / 10 ;
  pUSART->US_BRGR = brg;                                      // set baudrate generator
  pUSART->US_MR  = lcr_d | lcr_p | lcr_s;                     // DataBit, Parity, StopBit
  pUSART->US_CR  = AT91C_US_RSTRX  |                          // reset RX
                   AT91C_US_RSTTX  |                          // reset TX 
                   AT91C_US_RSTSTA;                           // reset  Status
  pUSART->US_CR  = AT91C_US_RXEN  | AT91C_US_TXEN;            // enable RX, TX

  ser_txRestart = 1;                                          // TX fifo is empty

  // Set up and enable the UART interrupt in the VIC
  AT91C_BASE_AIC->AIC_SMR[AT91C_ID_US0] = AT91C_AIC_SRCTYPE_INT_HIGH_LEVEL |
                                          AT91C_AIC_PRIOR_LOWEST;
  AT91C_BASE_AIC->AIC_SVR[AT91C_ID_US0] = (unsigned long) ser_irq_0;
  AT91C_BASE_AIC->AIC_IECR = (1 << AT91C_ID_US0);

  pUSART->US_IER = AT91C_US_RXRDY | AT91C_US_TXRDY;           // enable RX, TX interrupts
}

/*----------------------------------------------------------------------------
  read data from serial port
 *---------------------------------------------------------------------------*/
int ser_Read (char *buffer, const int *length) {
  int bytesToRead, bytesRead;
  
  // Read *length bytes, block if *bytes are not avaialable
  bytesToRead = *length;
  bytesRead   = bytesToRead;

  while (bytesToRead--) {
    while (SER_BUF_EMPTY(ser_in));                            // Block until data is available if none
    *buffer++ = SER_BUF_RD(ser_in);
  }
  return (bytesRead);  
}

/*----------------------------------------------------------------------------
  write data to the serial port
 *---------------------------------------------------------------------------*/
int ser_Write (const char *buffer, int *length) {
  int  bytesToWrite, bytesWritten;

  // Write *length bytes
  bytesToWrite = *length;
  bytesWritten = bytesToWrite;

  while (!SER_BUF_EMPTY(ser_out));                            // Block until space is available if none
  while (bytesToWrite) {
      SER_BUF_WR(ser_out, *buffer++);                         // Read Rx FIFO to buffer  
      bytesToWrite--;
  }     

  if (ser_txRestart) {
    ser_txRestart = 0;
    pUSART->US_IER = AT91C_US_TXRDY;                          // enable TX interrupt
  }

  return (bytesWritten); 
}

/*----------------------------------------------------------------------------
  check if character(s) are available at the serial interface
 *---------------------------------------------------------------------------*/
void ser_AvailChar (int *availChar) {

  *availChar = SER_BUF_COUNT(ser_in);

}

/*----------------------------------------------------------------------------
  read the line state of the serial port
 *---------------------------------------------------------------------------*/
void ser_LineState (unsigned int *lineState) {

  *lineState = ser_lineState;
  ser_lineState = 0;

}

/*----------------------------------------------------------------------------
  serial port 1 interrupt
 *---------------------------------------------------------------------------*/
static __irq void ser_irq_0 (void) { 
  volatile unsigned long iir;
  
  iir = pUSART->US_IMR & pUSART->US_CSR;   
  if ((iir & AT91C_US_RXRDY)) {                               // RXRDY interrupt
      SER_BUF_WR(ser_in,(unsigned char)pUSART->US_RHR);       // Read Rx FIFO to buffer  
  }
  if ((iir & AT91C_US_TXRDY)) {                               // TXRDY pending
	if (SER_BUF_COUNT(ser_out) != 0) {
      pUSART->US_THR = SER_BUF_RD(ser_out);                   // Write to the Tx FIFO
      ser_txRestart = 0;
    }
	else {
      pUSART->US_IDR = AT91C_US_TXRDY;
      ser_txRestart = 1;
	}
  }

  ser_lineState = pUSART->US_CSR;                             // update linestate

  AT91C_BASE_AIC->AIC_ICCR = (1 << AT91C_ID_US0);
  *AT91C_AIC_EOICR = 0;                                       // acknowledge interrupt
}


